<?php
require_once 'config/database.php';
require_once 'config/file_upload.php';

$pdo = conectarDB();

// Obtener categorías únicas para el select
$stmt_categorias = $pdo->query("SELECT DISTINCT categoria FROM repositorio ORDER BY categoria");
$categorias = $stmt_categorias->fetchAll(PDO::FETCH_COLUMN);

// Obtener autores únicos para el select
$stmt_autores = $pdo->query("SELECT DISTINCT autor FROM repositorio ORDER BY autor");
$autores = $stmt_autores->fetchAll(PDO::FETCH_COLUMN);

// Procesar filtros
$filtros = [];
$parametros = [];

if (isset($_GET['buscar']) && !empty($_GET['buscar'])) {
    $filtros[] = "(titulo LIKE ? OR descripcion LIKE ?)";
    $buscar = '%' . $_GET['buscar'] . '%';
    $parametros[] = $buscar;
    $parametros[] = $buscar;
}

if (isset($_GET['categoria']) && !empty($_GET['categoria'])) {
    $filtros[] = "categoria = ?";
    $parametros[] = $_GET['categoria'];
}

if (isset($_GET['autor']) && !empty($_GET['autor'])) {
    $filtros[] = "autor = ?";
    $parametros[] = $_GET['autor'];
}

// Construir consulta
$sql = "SELECT * FROM repositorio";
if (!empty($filtros)) {
    $sql .= " WHERE " . implode(" AND ", $filtros);
}
$sql .= " ORDER BY fecha_publicado DESC";

$stmt = $pdo->prepare($sql);
$stmt->execute($parametros);
$resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Buscar Repositorios</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 min-h-screen">
    <div class="container mx-auto px-4 py-8">
        <h1 class="text-3xl font-bold text-gray-800 mb-8">Buscar y Filtrar Repositorios</h1>
        
        <!-- Formulario de búsqueda -->
        <div class="bg-white rounded-lg shadow-md p-6 mb-8">
            <h2 class="text-xl font-semibold mb-4">Filtros de Búsqueda</h2>
            
            <form method="GET" class="space-y-4">
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <!-- Búsqueda por texto -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Buscar por título o descripción</label>
                        <input type="text" name="buscar" 
                               value="<?php echo htmlspecialchars($_GET['buscar'] ?? ''); ?>"
                               placeholder="Ingrese palabras clave..."
                               class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                    </div>
                    
                    <!-- Filtro por categoría -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Categoría</label>
                        <select name="categoria" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Todas las categorías</option>
                            <?php foreach ($categorias as $categoria): ?>
                                <option value="<?php echo htmlspecialchars($categoria); ?>" 
                                        <?php echo (isset($_GET['categoria']) && $_GET['categoria'] === $categoria) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($categoria); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- Filtro por autor -->
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Autor</label>
                        <select name="autor" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500">
                            <option value="">Todos los autores</option>
                            <?php foreach ($autores as $autor): ?>
                                <option value="<?php echo htmlspecialchars($autor); ?>" 
                                        <?php echo (isset($_GET['autor']) && $_GET['autor'] === $autor) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($autor); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="flex gap-2">
                    <button type="submit" 
                            class="bg-blue-500 hover:bg-blue-600 text-white px-6 py-2 rounded-md transition duration-200">
                        Buscar
                    </button>
                    <a href="buscar_repositorios.php" 
                       class="bg-gray-500 hover:bg-gray-600 text-white px-6 py-2 rounded-md transition duration-200">
                        Limpiar Filtros
                    </a>
                </div>
            </form>
        </div>
        
        <!-- Resultados -->
        <div class="bg-white rounded-lg shadow-md">
            <div class="p-6 border-b">
                <h2 class="text-xl font-semibold">
                    Resultados de la búsqueda
                    <?php if (!empty($resultados)): ?>
                        <span class="text-sm font-normal text-gray-600">(<?php echo count($resultados); ?> encontrados)</span>
                    <?php endif; ?>
                </h2>
            </div>
            
            <?php if (empty($resultados)): ?>
                <div class="p-6 text-center text-gray-500">
                    <?php if (isset($_GET['buscar']) || isset($_GET['categoria']) || isset($_GET['autor'])): ?>
                        No se encontraron repositorios con los criterios especificados.
                    <?php else: ?>
                        No hay repositorios disponibles. Utilice los filtros para buscar.
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <div class="divide-y divide-gray-200">
                    <?php foreach ($resultados as $repo): ?>
                        <div class="p-6 hover:bg-gray-50 transition duration-200">
                            <div class="flex gap-4">
                                <!-- Thumbnail -->
                                <div class="flex-shrink-0">
                                    <?php if ($repo['thumbnails'] && file_exists(THUMBNAIL_DIR . $repo['thumbnails'])): ?>
                                        <img src="<?php echo THUMBNAIL_DIR . htmlspecialchars($repo['thumbnails']); ?>" 
                                             alt="Vista previa del documento" 
                                             class="w-24 h-32 object-cover rounded-lg border border-gray-200 shadow-sm">
                                    <?php else: ?>
                                        <div class="w-24 h-32 bg-gray-200 rounded-lg border border-gray-200 flex items-center justify-center">
                                            <span class="text-gray-500 text-xs">PDF</span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                
                                <!-- Contenido -->
                                <div class="flex-1">
                                    <div class="flex items-center gap-2 mb-2">
                                        <span class="inline-block bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">
                                            <?php echo htmlspecialchars($repo['categoria']); ?>
                                        </span>
                                        <span class="text-sm text-gray-500">
                                            <?php echo date('d/m/Y', strtotime($repo['fecha_publicado'])); ?>
                                        </span>
                                    </div>
                                    
                                    <h3 class="text-lg font-semibold text-gray-900 mb-2">
                                        <a href="detalle_repositorio.php?id=<?php echo $repo['id']; ?>" 
                                           class="hover:text-blue-600 transition duration-200">
                                            <?php echo htmlspecialchars($repo['titulo']); ?>
                                        </a>
                                    </h3>
                                    
                                    <p class="text-gray-600 mb-2">
                                        <strong>Autor:</strong> <?php echo htmlspecialchars($repo['autor']); ?>
                                    </p>
                                    
                                    <?php if (!empty($repo['descripcion'])): ?>
                                        <p class="text-gray-700 mb-3">
                                            <?php echo htmlspecialchars(substr($repo['descripcion'], 0, 200)); ?>
                                            <?php if (strlen($repo['descripcion']) > 200): ?>...<?php endif; ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                                
                                <!-- Acciones -->
                                <div class="flex flex-col gap-2 ml-4">
                                    <a href="detalle_repositorio.php?id=<?php echo $repo['id']; ?>" 
                                       class="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-md text-sm transition duration-200">
                                        Ver Detalle
                                    </a>
                                    <?php if (!empty($repo['url_documento']) && file_exists(UPLOAD_DIR . $repo['url_documento'])): ?>
                                        <a href="<?php echo UPLOAD_DIR . htmlspecialchars($repo['url_documento']); ?>" 
                                           target="_blank"
                                           class="bg-green-500 hover:bg-green-600 text-white px-4 py-2 rounded-md text-sm transition duration-200">
                                            Descargar
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Enlaces a otras páginas -->
        <div class="mt-8 flex gap-4">
            <a href="gestion_repositorios.php" 
               class="bg-purple-500 hover:bg-purple-600 text-white px-6 py-2 rounded-md transition duration-200">
                Gestión de Repositorios
            </a>
        </div>
    </div>
</body>
</html>
