<?php
require_once 'config/database.php';
require_once 'config/file_upload.php';

// Verificar que se proporciona un ID
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: buscar_repositorios.php');
    exit;
}

$id = $_GET['id'];
$pdo = conectarDB();

// Obtener el repositorio
$stmt = $pdo->prepare("SELECT * FROM repositorio WHERE id = ?");
$stmt->execute([$id]);
$repositorio = $stmt->fetch(PDO::FETCH_ASSOC);

// Si no existe el repositorio, redirigir
if (!$repositorio) {
    header('Location: buscar_repositorios.php');
    exit;
}

// Función para descargar archivo
if (isset($_GET['descargar']) && $_GET['descargar'] === 'true') {
    $archivo_path = UPLOAD_DIR . $repositorio['url_documento'];
    
    // Verificar si el archivo existe
    if (file_exists($archivo_path)) {
        // Obtener información del archivo
        $nombre_archivo = basename($archivo_path);
        $tamaño_archivo = filesize($archivo_path);
        $tipo_mime = mime_content_type($archivo_path);
        
        // Configurar headers para descarga
        header('Content-Type: ' . $tipo_mime);
        header('Content-Disposition: attachment; filename="' . $nombre_archivo . '"');
        header('Content-Length: ' . $tamaño_archivo);
        header('Cache-Control: no-cache, must-revalidate');
        header('Pragma: no-cache');
        
        // Leer y enviar el archivo
        readfile($archivo_path);
        exit;
    } else {
        $error_descarga = "El archivo no se encuentra disponible para descarga.";
    }
}

// Obtener repositorios relacionados (misma categoría)
$stmt_relacionados = $pdo->prepare("SELECT id, titulo, autor FROM repositorio WHERE categoria = ? AND id != ? ORDER BY fecha_publicado DESC LIMIT 3");
$stmt_relacionados->execute([$repositorio['categoria'], $id]);
$repositorios_relacionados = $stmt_relacionados->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($repositorio['titulo']); ?> - Detalle del Repositorio</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-100 min-h-screen">
    <div class="container mx-auto px-4 py-8">
        <!-- Navegación -->
        <div class="mb-6">
            <a href="buscar_repositorios.php" 
               class="inline-flex items-center text-blue-600 hover:text-blue-800 transition duration-200">
                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path>
                </svg>
                Volver a la búsqueda
            </a>
        </div>
        
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
            <!-- Contenido principal -->
            <div class="lg:col-span-2">
                <div class="bg-white rounded-lg shadow-md p-6 mb-6">
                    <!-- Header del repositorio -->
                    <div class="mb-6">
                        <div class="flex items-center gap-2 mb-3">
                            <span class="inline-block bg-blue-100 text-blue-800 text-sm px-3 py-1 rounded-full font-medium">
                                <?php echo htmlspecialchars($repositorio['categoria']); ?>
                            </span>
                            <span class="text-sm text-gray-500">
                                Publicado el <?php echo date('d/m/Y', strtotime($repositorio['fecha_publicado'])); ?>
                            </span>
                        </div>
                        
                        <h1 class="text-3xl font-bold text-gray-900 mb-4">
                            <?php echo htmlspecialchars($repositorio['titulo']); ?>
                        </h1>
                        
                        <div class="flex items-center text-gray-600 mb-4">
                            <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"></path>
                            </svg>
                            <span class="font-medium">Autor:</span>
                            <span class="ml-2"><?php echo htmlspecialchars($repositorio['autor']); ?></span>
                        </div>
                    </div>
                    
                    <!-- Descripción -->
                    <?php if (!empty($repositorio['descripcion'])): ?>
                        <div class="mb-6">
                            <h2 class="text-xl font-semibold text-gray-900 mb-3">Descripción</h2>
                            <div class="prose max-w-none text-gray-700">
                                <?php echo nl2br(htmlspecialchars($repositorio['descripcion'])); ?>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Vista previa del documento -->
                    <?php if ($repositorio['thumbnails'] && file_exists(THUMBNAIL_DIR . $repositorio['thumbnails'])): ?>
                        <div class="mb-6">
                            <h2 class="text-xl font-semibold text-gray-900 mb-3">Vista Previa del Documento</h2>
                            <div class="bg-gray-50 rounded-lg p-4">
                                <div class="flex justify-center">
                                    <img src="<?php echo THUMBNAIL_DIR . htmlspecialchars($repositorio['thumbnails']); ?>" 
                                         alt="Vista previa del documento" 
                                         class="max-w-full h-auto rounded-lg shadow-md border border-gray-200">
                                </div>
                                <p class="text-center text-sm text-gray-500 mt-2">
                                    Primera página del documento
                                </p>
                            </div>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Información del archivo -->
                    <div class="mb-6">
                        <h2 class="text-xl font-semibold text-gray-900 mb-3">Información del Documento</h2>
                        <div class="bg-gray-50 rounded-lg p-4">
                            <div class="flex items-center justify-between">
                                <div class="flex items-center">
                                    <svg class="w-8 h-8 text-gray-400 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                    </svg>
                                    <div>
                                        <p class="font-medium text-gray-900">
                                            <?php echo htmlspecialchars($repositorio['url_documento']); ?>
                                        </p>
                                        <p class="text-sm text-gray-500">
                                            Archivo PDF
                                        </p>
                                    </div>
                                </div>
                                
                                <div class="flex gap-2">
                                    <?php if (file_exists(UPLOAD_DIR . $repositorio['url_documento'])): ?>
                                        <a href="detalle_repositorio.php?id=<?php echo $id; ?>&descargar=true" 
                                           class="bg-green-500 hover:bg-green-600 text-white px-4 py-2 rounded-md transition duration-200 flex items-center">
                                            <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                                            </svg>
                                            Descargar
                                        </a>
                                    <?php else: ?>
                                        <span class="bg-red-100 text-red-800 px-4 py-2 rounded-md text-sm">
                                            Archivo no disponible
                                        </span>
                                    <?php endif; ?>
                                    
                                    <a href="<?php echo UPLOAD_DIR . htmlspecialchars($repositorio['url_documento']); ?>" 
                                       target="_blank"
                                       class="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-md transition duration-200 flex items-center">
                                        <svg class="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"></path>
                                        </svg>
                                        Abrir
                                    </a>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Error de descarga -->
                    <?php if (isset($error_descarga)): ?>
                        <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                            <?php echo htmlspecialchars($error_descarga); ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Sidebar -->
            <div class="lg:col-span-1">
                <!-- Repositorios relacionados -->
                <?php if (!empty($repositorios_relacionados)): ?>
                    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
                        <h3 class="text-lg font-semibold text-gray-900 mb-4">
                            Repositorios Relacionados
                        </h3>
                        <div class="space-y-3">
                            <?php foreach ($repositorios_relacionados as $relacionado): ?>
                                <div class="border-l-4 border-blue-500 pl-4 py-2">
                                    <a href="detalle_repositorio.php?id=<?php echo $relacionado['id']; ?>" 
                                       class="block hover:text-blue-600 transition duration-200">
                                        <h4 class="font-medium text-gray-900 text-sm">
                                            <?php echo htmlspecialchars($relacionado['titulo']); ?>
                                        </h4>
                                        <p class="text-xs text-gray-500 mt-1">
                                            <?php echo htmlspecialchars($relacionado['autor']); ?>
                                        </p>
                                    </a>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                <?php endif; ?>
                
                <!-- Acciones rápidas -->
                <div class="bg-white rounded-lg shadow-md p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Acciones</h3>
                    <div class="space-y-2">
                        <a href="gestion_repositorios.php?editar=<?php echo $id; ?>" 
                           class="block w-full bg-yellow-500 hover:bg-yellow-600 text-white text-center px-4 py-2 rounded-md transition duration-200">
                            Editar Repositorio
                        </a>
                        <a href="buscar_repositorios.php?categoria=<?php echo urlencode($repositorio['categoria']); ?>" 
                           class="block w-full bg-blue-500 hover:bg-blue-600 text-white text-center px-4 py-2 rounded-md transition duration-200">
                            Ver Más de <?php echo htmlspecialchars($repositorio['categoria']); ?>
                        </a>
                        <a href="buscar_repositorios.php?autor=<?php echo urlencode($repositorio['autor']); ?>" 
                           class="block w-full bg-purple-500 hover:bg-purple-600 text-white text-center px-4 py-2 rounded-md transition duration-200">
                            Ver Más de <?php echo htmlspecialchars($repositorio['autor']); ?>
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
