<?php
// Configuración para carga de archivos
define('UPLOAD_DIR', 'uploads/documents/');
define('THUMBNAIL_DIR', 'uploads/thumbnails/');
define('MAX_FILE_SIZE', 50 * 1024 * 1024); // 50 MB en bytes
define('ALLOWED_EXTENSIONS', ['pdf']);

// Crear directorios si no existen
function crearDirectorios() {
    if (!file_exists(UPLOAD_DIR)) {
        mkdir(UPLOAD_DIR, 0755, true);
    }
    if (!file_exists(THUMBNAIL_DIR)) {
        mkdir(THUMBNAIL_DIR, 0755, true);
    }
}

// Validar archivo PDF
function validarArchivoPDF($archivo) {
    $errores = [];
    
    // Verificar si se subió un archivo
    if (!isset($archivo) || $archivo['error'] !== UPLOAD_ERR_OK) {
        $errores[] = 'Error al subir el archivo';
        return $errores;
    }
    
    // Verificar tamaño
    if ($archivo['size'] > MAX_FILE_SIZE) {
        $errores[] = 'El archivo es demasiado grande. Máximo permitido: 50MB';
    }
    
    // Verificar extensión
    $extension = strtolower(pathinfo($archivo['name'], PATHINFO_EXTENSION));
    if (!in_array($extension, ALLOWED_EXTENSIONS)) {
        $errores[] = 'Solo se permiten archivos PDF';
    }
    
    // Verificar tipo MIME
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $archivo['tmp_name']);
    finfo_close($finfo);
    
    if ($mimeType !== 'application/pdf') {
        $errores[] = 'El archivo no es un PDF válido';
    }
    
    return $errores;
}

// Generar nombre único para archivo
function generarNombreArchivo($nombreOriginal) {
    $extension = pathinfo($nombreOriginal, PATHINFO_EXTENSION);
    $nombreBase = pathinfo($nombreOriginal, PATHINFO_FILENAME);
    $nombreBase = preg_replace('/[^a-zA-Z0-9_-]/', '_', $nombreBase);
    return $nombreBase . '_' . uniqid() . '.' . $extension;
}

// Subir archivo PDF
function subirArchivoPDF($archivo) {
    crearDirectorios();
    
    $errores = validarArchivoPDF($archivo);
    if (!empty($errores)) {
        return ['success' => false, 'errores' => $errores];
    }
    
    $nombreArchivo = generarNombreArchivo($archivo['name']);
    $rutaDestino = UPLOAD_DIR . $nombreArchivo;
    
    if (move_uploaded_file($archivo['tmp_name'], $rutaDestino)) {
        return [
            'success' => true, 
            'archivo' => $nombreArchivo,
            'ruta' => $rutaDestino
        ];
    } else {
        return ['success' => false, 'errores' => ['Error al mover el archivo']];
    }
}

// Generar thumbnail de PDF (versión simplificada y confiable)
function generarThumbnailPDF($rutaPDF, $nombreArchivo) {
    crearDirectorios();
    
    $nombreThumbnail = pathinfo($nombreArchivo, PATHINFO_FILENAME) . '_thumb.jpg';
    $rutaThumbnail = THUMBNAIL_DIR . $nombreThumbnail;
    
    // Método 1: ImageMagick (más confiable)
    if (extension_loaded('imagick') && class_exists('Imagick')) {
        try {
            $imagick = new Imagick();
            $imagick->setResolution(150, 150);
            $imagick->readImage($rutaPDF . '[0]'); // Primera página
            $imagick->setImageFormat('jpeg');
            $imagick->setImageCompressionQuality(85);
            $imagick->thumbnailImage(200, 200, true, true);
            $imagick->writeImage($rutaThumbnail);
            $imagick->clear();
            $imagick->destroy();
            
            if (file_exists($rutaThumbnail)) {
                return $nombreThumbnail;
            }
        } catch (Exception $e) {
            error_log("Error generando thumbnail con ImageMagick: " . $e->getMessage());
        }
    }
    
    // Método 2: Fallback - crear placeholder mejorado con GD
    if (extension_loaded('gd')) {
        $imagen = imagecreatetruecolor(200, 200);
        
        // Colores
        $fondo = imagecolorallocate($imagen, 245, 245, 245);
        $borde = imagecolorallocate($imagen, 200, 200, 200);
        $texto = imagecolorallocate($imagen, 100, 100, 100);
        $titulo = imagecolorallocate($imagen, 50, 50, 50);
        $pdfIcon = imagecolorallocate($imagen, 220, 38, 38); // Rojo
        $blanco = imagecolorallocate($imagen, 255, 255, 255);
        
        // Fondo
        imagefill($imagen, 0, 0, $fondo);
        
        // Borde
        imagerectangle($imagen, 0, 0, 199, 199, $borde);
        
        // Icono de PDF (rectángulo con esquina doblada)
        imagefilledrectangle($imagen, 60, 50, 140, 120, $pdfIcon);
        imagefilledrectangle($imagen, 120, 50, 140, 70, $fondo);
        
        // Texto "PDF"
        imagestring($imagen, 4, 80, 90, 'PDF', $blanco);
        
        // Nombre del archivo (truncado)
        $nombreCorto = substr(pathinfo($nombreArchivo, PATHINFO_FILENAME), 0, 12);
        if (strlen($nombreCorto) < strlen(pathinfo($nombreArchivo, PATHINFO_FILENAME))) {
            $nombreCorto .= '...';
        }
        imagestring($imagen, 2, 10, 130, $nombreCorto, $titulo);
        
        // Tamaño del archivo
        if (file_exists($rutaPDF)) {
            $tamaño = filesize($rutaPDF);
            $tamañoTexto = formatBytes($tamaño);
            imagestring($imagen, 2, 10, 150, $tamañoTexto, $texto);
        }
        
        // Fecha
        imagestring($imagen, 2, 10, 170, date('d/m/Y'), $texto);
        
        imagejpeg($imagen, $rutaThumbnail, 85);
        imagedestroy($imagen);
        
        return $nombreThumbnail;
    }
    
    return null;
}

// Función auxiliar para verificar si un comando está disponible
function isCommandAvailable($command) {
    $return = shell_exec(sprintf("which %s", escapeshellarg($command)));
    return !empty($return);
}

// Función auxiliar para eliminar directorio recursivamente
function deleteDirectory($dir) {
    if (!is_dir($dir)) {
        return false;
    }
    
    $files = array_diff(scandir($dir), array('.', '..'));
    foreach ($files as $file) {
        $path = $dir . $file;
        if (is_dir($path)) {
            deleteDirectory($path . '/');
        } else {
            unlink($path);
        }
    }
    return rmdir($dir);
}

// Función auxiliar para formatear bytes
function formatBytes($bytes, $precision = 2) {
    $units = array('B', 'KB', 'MB', 'GB', 'TB');
    
    for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
        $bytes /= 1024;
    }
    
    return round($bytes, $precision) . ' ' . $units[$i];
}

// Eliminar archivo y su thumbnail
function eliminarArchivo($nombreArchivo, $nombreThumbnail = null) {
    $rutaArchivo = UPLOAD_DIR . $nombreArchivo;
    if (file_exists($rutaArchivo)) {
        unlink($rutaArchivo);
    }
    
    if ($nombreThumbnail) {
        $rutaThumbnail = THUMBNAIL_DIR . $nombreThumbnail;
        if (file_exists($rutaThumbnail)) {
            unlink($rutaThumbnail);
        }
    }
}

// Obtener información del archivo
function obtenerInfoArchivo($nombreArchivo) {
    $rutaArchivo = UPLOAD_DIR . $nombreArchivo;
    if (!file_exists($rutaArchivo)) {
        return null;
    }
    
    return [
        'nombre' => $nombreArchivo,
        'ruta' => $rutaArchivo,
        'url' => UPLOAD_DIR . $nombreArchivo,
        'tamaño' => filesize($rutaArchivo),
        'fecha' => filemtime($rutaArchivo)
    ];
}
?>
